<?php
// install_smart.php — FK-safe installer
// - Detects f9632e5_pos.products.id type/charset/collation/engine
// - Ensures products is InnoDB and indexed
// - Creates tables with matching column types for product_id
// - Adds FKs without error 150
require __DIR__.'/db.php';
header('Content-Type: text/plain; charset=utf-8');

try {
  $pdo = pdo();
  $db  = env('DB_NAME');
  echo "Connected to DB: $db\n\n";

  // 1) Verify products table exists
  $stmt = $pdo->prepare("SELECT TABLE_NAME, ENGINE FROM information_schema.TABLES WHERE TABLE_SCHEMA=? AND TABLE_NAME='products'");
  $stmt->execute([$db]);
  $row = $stmt->fetch();
  if (!$row) throw new Exception("Table $db.products not found. Create/import your products table first.");
  $engine = $row['ENGINE'];
  echo "products ENGINE: $engine\n";

  if (strtoupper($engine) !== 'INNODB') {
    echo "Converting products to InnoDB...\n";
    $pdo->exec("ALTER TABLE `products` ENGINE=InnoDB");
  }

  // 2) Get products.id definition
  $col = $pdo->prepare("SELECT DATA_TYPE, COLUMN_TYPE, CHARACTER_SET_NAME, COLLATION_NAME, COLUMN_KEY FROM information_schema.COLUMNS WHERE TABLE_SCHEMA=? AND TABLE_NAME='products' AND COLUMN_NAME='id'");
  $col->execute([$db]);
  $c = $col->fetch();
  if (!$c) throw new Exception("Column products.id not found.");
  echo "products.id => DATA_TYPE={$c['DATA_TYPE']}, COLUMN_TYPE={$c['COLUMN_TYPE']}, CHARSET={$c['CHARACTER_SET_NAME']}, COLLATION={$c['COLLATION_NAME']}, KEY={$c['COLUMN_KEY']}\n";

  // 3) Ensure id is indexed (PK/UNIQUE or at least normal index)
  if (!in_array($c['COLUMN_KEY'], ['PRI','UNI'])) {
    echo "Adding index on products.id ...\n";
    // avoid duplicate index error if one already exists
    $idx = $pdo->query("SHOW INDEX FROM `products` WHERE Column_name='id'")->fetch();
    if (!$idx) $pdo->exec("ALTER TABLE `products` ADD INDEX `idx_products_id` (`id`)");
  }

  // 4) Helper to produce a matching column definition
  function matching_col_def($c){
    $dt = strtolower($c['DATA_TYPE']);
    $ctype = $c['COLUMN_TYPE']; // e.g., int(11) unsigned, varchar(64)
    if (in_array($dt, ['char','varchar','binary','varbinary'])) {
      $charset = $c['CHARACTER_SET_NAME'];
      $coll = $c['COLLATION_NAME'];
      $len = preg_replace('/[^0-9]/', '', $ctype);
      if (!$len) $len = 191; // safe default
      $cs = $charset ? " CHARACTER SET $charset" : "";
      $co = $coll ? " COLLATE $coll" : "";
      return "VARCHAR($len)$cs$co";
    }
    // text/blob cannot be FK targets: but COLUMN_TYPE will include text; we bail
    if (in_array($dt, ['text','tinytext','mediumtext','longtext','blob','tinyblob','mediumblob','longblob'])) {
      throw new Exception("products.id is $dt which cannot be safely referenced by FK. Change products.id to INT/VARCHAR and index it.");
    }
    // numeric: use COLUMN_TYPE as-is
    return strtoupper($ctype); // e.g., INT(11) UNSIGNED
  }

  $prodIdDef = matching_col_def($c);
  echo "Matching product_id column: $prodIdDef\n";

  // 5) Create required tables if not exist (InnoDB)
  $pdo->exec("CREATE TABLE IF NOT EXISTS staff (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    role ENUM('kitchen','front') NOT NULL,
    pin VARCHAR(20) NOT NULL,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
  ) ENGINE=InnoDB");

  $pdo->exec("CREATE TABLE IF NOT EXISTS productions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_id $prodIdDef NOT NULL,
    qty INT NOT NULL,
    staff_id INT NOT NULL,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_prod_product_id (product_id),
    INDEX idx_prod_staff_id (staff_id),
    CONSTRAINT fk_productions_staff FOREIGN KEY (staff_id) REFERENCES staff(id)
  ) ENGINE=InnoDB");

  $pdo->exec("CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    placed_by INT NOT NULL,
    status ENUM('pending','accepted','preparing','ready') NOT NULL DEFAULT 'pending',
    accepted_by INT NULL,
    accepted_at DATETIME NULL,
    updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_orders_placed_by (placed_by),
    INDEX idx_orders_accepted_by (accepted_by),
    CONSTRAINT fk_orders_placed_by FOREIGN KEY (placed_by) REFERENCES staff(id),
    CONSTRAINT fk_orders_accepted_by FOREIGN KEY (accepted_by) REFERENCES staff(id)
  ) ENGINE=InnoDB");

  $pdo->exec("CREATE TABLE IF NOT EXISTS order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id $prodIdDef NOT NULL,
    qty INT NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    INDEX idx_order_items_order_id (order_id),
    INDEX idx_order_items_product_id (product_id),
    CONSTRAINT fk_order_items_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE
  ) ENGINE=InnoDB");

  $pdo->exec("CREATE TABLE IF NOT EXISTS order_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    event VARCHAR(32) NOT NULL,
    staff_id INT NULL,
    at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_order_events_order_id (order_id),
    INDEX idx_order_events_staff_id (staff_id),
    CONSTRAINT fk_events_order FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    CONSTRAINT fk_events_staff FOREIGN KEY (staff_id) REFERENCES staff(id)
  ) ENGINE=InnoDB");

  $pdo->exec("CREATE TABLE IF NOT EXISTS bumps (
    id TINYINT PRIMARY KEY,
    version BIGINT NOT NULL DEFAULT 0
  ) ENGINE=InnoDB");
  $pdo->exec("INSERT INTO bumps (id, version) VALUES (1,0) ON DUPLICATE KEY UPDATE version=version");

  // 6) Add FKs to products only after both child tables exist and types match
  // Use try..catch to avoid fatal if it already exists
  try {
    $pdo->exec("ALTER TABLE productions
      ADD CONSTRAINT fk_productions_product FOREIGN KEY (product_id) REFERENCES products(id)");
    echo "FK: productions.product_id -> products.id added.\n";
  } catch (Throwable $e) {
    echo "Note: productions FK skipped/exists: ".$e->getMessage()."\n";
  }

  try {
    $pdo->exec("ALTER TABLE order_items
      ADD CONSTRAINT fk_order_items_product FOREIGN KEY (product_id) REFERENCES products(id)");
    echo "FK: order_items.product_id -> products.id added.\n";
  } catch (Throwable $e) {
    echo "Note: order_items FK skipped/exists: ".$e->getMessage()."\n";
  }

  echo "\nDone.\n";

} catch (Throwable $e) {
  http_response_code(500);
  echo "ERROR: ".$e->getMessage()."\n";
}
